--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Dataflow architecture of the decoder module
--  
--  (file decoder-dataflow.vhd)
--------------------------------------------------------------------------

architecture dataflow of decoder is

  signal all_select : std_logic_vector(0 to 4);
  alias  sel_sysrom : std_logic is all_select(0);
  alias  sel_userom : std_logic is all_select(1);
  alias  sel_timer  : std_logic is all_select(2);
  alias  sel_sysram : std_logic is all_select(3);
  alias  sel_useram : std_logic is all_select(4);

  signal sel_sysram_vec: dlx_nibble;
  signal sel_useram_vec: dlx_nibble;
  signal sel_sysram_rdy: std_logic;
  signal sel_useram_rdy: std_logic;
  signal int_or_enab: std_logic;
  
begin
  --------------------------- 
  -- functional part
  ---------------------------

  -- decode most significant address nibble to select component
  all_select <=
    "10000" when enable = "1111" and a_bus_hn = "0000" else   -- system rom is selected
    "01000" when enable = "1111" and a_bus_hn = "1000" else   -- user rom is selected
    "00100" when (enable = "1111" or enable = "1100" or
                 enable = "0011" or enable = "1000" or
 		 enable = "0100" or enable = "0010" or
		 enable = "0001") and a_bus_hn = "0111" else  -- timer is selected
    "00010" when (enable = "1111" or enable = "1100" or
                 enable = "0011" or enable = "1000" or
 		 enable = "0100" or enable = "0010" or
		 enable = "0001") and a_bus_hn = "0001" else  -- system ram is selected
    "00001" when (enable = "1111" or enable = "1100" or
                 enable = "0011" or enable = "1000" or
 		 enable = "0100" or enable = "0010" or
		 enable = "0001") and a_bus_hn = "1001" else  -- system ram is selected
    		 "00000" when enable = "0000" else            -- nothing selected
    "XXXXX";                                                  -- undefined inputs !
    
  -- decode enable lines for the two ram banks (byte organized)
  sel_sysram_vec <= enable when sel_sysram = '1' else
                    "0000";
  sel_useram_vec <= enable when sel_useram = '1' else
                    "0000";

  -- connect internal signals to ports 
  enab_sysrom <= sel_sysrom after tpd;
  enab_userom <= sel_userom after tpd;
  enab_timer  <= sel_timer after tpd;
  enab_sysram <= sel_sysram_vec after tpd;
  enab_useram <= sel_useram_vec after tpd;  
  ram_ready   <= sel_sysram_rdy after tpd when sel_sysram = '1' else
                 sel_useram_rdy after tpd when sel_useram = '1' else
		 'Z';
  
  -- select lines to generate ram_ready signal
  -- when ram is selected
  
  sel_sysram_rdy <= (in_sysram_rdy(0) and in_sysram_rdy(1)
                     and in_sysram_rdy(2) and in_sysram_rdy(3))
                      when enable = "1111" and sel_sysram = '1' else
		    (in_sysram_rdy(0) and in_sysram_rdy(1))
 		      when enable = "1100" and sel_sysram = '1' else
		    (in_sysram_rdy(2) and in_sysram_rdy(3))
 		      when enable = "0011" and sel_sysram = '1' else
   		    in_sysram_rdy(0) when enable = "1000" and sel_sysram = '1' else
   		    in_sysram_rdy(1) when enable = "0100" and sel_sysram = '1' else
   		    in_sysram_rdy(2) when enable = "0010" and sel_sysram = '1' else
   		    in_sysram_rdy(3) when enable = "0001" and sel_sysram = '1' else
		    'Z';

  sel_useram_rdy <= (in_useram_rdy(0) and in_useram_rdy(1)
                     and in_useram_rdy(2) and in_useram_rdy(3))
                      when enable = "1111" and sel_useram = '1' else
		    (in_useram_rdy(0) and in_useram_rdy(1))
 		      when enable = "1100" and sel_useram = '1' else
		    (in_useram_rdy(2) and in_useram_rdy(3))
 		      when enable = "0011" and sel_useram = '1' else
   		    in_useram_rdy(0) when enable = "1000" and sel_useram = '1' else
   		    in_useram_rdy(1) when enable = "0100" and sel_useram = '1' else
   		    in_useram_rdy(2) when enable = "0010" and sel_useram = '1' else
   		    in_useram_rdy(3) when enable = "0001" and sel_useram = '1' else
		    'Z';		 
		 
  --------------------------- 
  -- debugging part
  ---------------------------
  watch1: process(sel_sysrom)
    variable L : line;
  begin
    if MONITOR and sel_sysrom ='1' then
      write(L, string'("decoder:  System ROM enabled"));
      writeline(output, L);
    end if;
  end process;

  watch2: process(sel_userom)
    variable L : line;
  begin
    if MONITOR and sel_userom ='1' then
      write(L, string'("decoder:  User ROM enabled"));
      writeline(output, L);
    end if;
  end process;
  
  watch3: process(sel_timer)
    variable L : line;
  begin
    if MONITOR and sel_timer ='1' then
      write(L, string'("decoder:  Timer enabled"));
      writeline(output, L);
    end if;
  end process;

  watch4: process(sel_sysram)
    variable L : line;
  begin
    if MONITOR and sel_sysram = '1' then
      write(L, string'("decoder:  System RAM enabled"));
      writeline(output, L);
    end if;
  end process;
  
  watch5: process(sel_useram)
    variable L : line;
  begin
    if MONITOR and sel_useram = '1' then
      write(L, string'("decoder:  User RAM enabled"));
      writeline(output, L);
    end if;
  end process;

  watch6: process(sel_sysram_rdy)
    variable L : line;
  begin
    if MONITOR and sel_sysram_rdy = '1' then
      write(L, string'("decoder:  System RAM ready"));
      writeline(output, L);
    end if;
  end process;

  watch7: process(sel_useram_rdy)
    variable L : line;
  begin
    if MONITOR and sel_useram_rdy ='1' then
      write(L, string'("decoder:  User RAM ready"));
      writeline(output, L);
    end if;
  end process;

end dataflow;




