--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Structural architecture of the DLX processor
--  (only signed instructions)
--  
--  (file dlx-structural.vhd)
--------------------------------------------------------------------------

architecture structural of dlx is

  component core
    port (a_bus  : out dlx_address;
       	  d_bus_in : in dlx_word;
	  d_bus_out: out dlx_word;
          enable : out dlx_nibble;
	  rw     : out std_logic;
	  error  : out std_logic;
	  ready  : in std_logic;
	  reset  : in std_logic;             -- asynchronous reset 
	  halt   : in std_logic;             -- freeze of processor state
	  intrpt : in dlx_nibble;            -- interupt signals (maskable)
          phi1   : in std_logic;             -- 2 phase non overlapping clock
	  phi2   : in std_logic;
	  pad_out_en : out std_logic;        -- output pads enable (0 = out, 1 = tri)
	  pad_io_sw  : out std_logic);       --io pads switch (0 = output, 1 = input)
  end component;

  component pad_io
    port (io_sel : in std_logic;              -- select input/output ('0' = out)
	  pad    : inout std_logic;           -- pad out
	  pad2core : out std_logic;           -- from pad to core
	  core2pad : in  std_logic);          -- from core to pad
  end component;

  component pad_out
    port (enable : in std_logic;              -- enable output ('0' = out)
	  pad    : out std_logic;             -- pad out
	  core2pad : in  std_logic);          -- from core to pad
  end component;

  component pad_in
    port (pad      : in std_logic;            -- pad input
	  pad2core : out std_logic);          -- from pad to core
  end component;

  --
  -- signals from/to pads
  --
  signal pad_data_in  : dlx_word;              -- data in from pads
  signal pad_data_out : dlx_word;              -- data_out to pads
  signal pad_addr_out : dlx_word;              -- addr. out to pads
  signal phi1_in   : std_logic;                -- clock input from pad
  signal phi2_in   : std_logic;                -- clock input from pad
  signal reset_in  : std_logic;                -- reset input from pad
  signal halt_in   : std_logic;                -- halt input from pad
  signal ready_in  : std_logic;                -- ready input from pad
  signal intrpt_in : dlx_nibble;               -- interrupt input from pad
  signal rw_out : std_logic;
  signal error_out : std_logic;
  signal enable_out : dlx_nibble;
  signal pad_out_en : std_logic;
  signal pad_io_sw : std_logic;
  
begin       

  the_core : core
    port map (a_bus => pad_addr_out,
              d_bus_in => pad_data_in,
 	      d_bus_out => pad_data_out,
              enable => enable_out,
 	      rw => rw_out,
 	      error => error_out,
 	      ready => ready_in,
 	      reset => reset_in,
 	      halt => halt_in,
              intrpt => intrpt_in,
	      phi1 => phi1_in,
	      phi2 => phi2_in,
	      pad_out_en => pad_out_en,
	      pad_io_sw => pad_io_sw);
	      
  --
  -- input pad cells
  --
  reset_pad : pad_in
    port map (pad => reset, pad2core => reset_in);
      
  halt_pad : pad_in
    port map (pad => halt, pad2core => halt_in);
      
  phi1_pad : pad_in
    port map (pad => phi1, pad2core => phi1_in);

  phi2_pad : pad_in
    port map (pad => phi2, pad2core => phi2_in);

  ready_pad : pad_in
    port map (pad => ready, pad2core =>  ready_in);
      
  interrupt_pads : for i in 0 to 3 generate
    intrpt_pad : pad_in
      port map (pad => intrpt(i), pad2core => intrpt_in(i));
  end generate;
  --
  -- input/output pad cells
  --
  data_pads : for i in 0 to 31 generate
    d_pad : pad_io
      port map (io_sel => pad_io_sw,
	        pad => d_bus(i),
	        pad2core => pad_data_in(i),
	        core2pad => pad_data_out(i));
  end generate;
  --
  -- output pad cells
  --  
  address_pads : for i in 0 to 31 generate
    a_pad : pad_out
      port map (enable => pad_out_en,
	        pad => a_bus(i),
	        core2pad => pad_addr_out(31-i));
  end generate;
  
  enable_pads : for i in 0 to 3 generate
    en_pad : pad_out
      port map (enable => pad_out_en,
                pad => enable(i),
	        core2pad => enable_out(i));
  end generate;

  rw_pad: pad_out
    port map (enable => pad_out_en,
              pad => rw,
              core2pad => rw_out);  

  error_pad: pad_out
    port map (enable => pad_out_en,
              pad => error,
              core2pad => error_out);  

end structural;




