--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  behavioural architecture of the DLX processor
--  (first implementation step of the behaviour)
--  
--  (file dlx-behaviour_1.vhd)
--------------------------------------------------------------------------

library IEEE_EXTD;
use IEEE_EXTD.stdl1164_vector_arithmetic.all,
    IEEE_EXTD.stdl1164_extended.all;

use STD.textio.all;

use WORK.dlx_instructions.all,
    WORK.behav_procedures_123.all;
  
architecture behaviour_1 of dlx is

begin                    -- behaviour
  
  interpreter: process
    --
    -- the following variables correspond to DLXS registers or
    -- bits or address fields
    --
    variable GP_REG : dlx_word_array(reg_index);
    variable PC  : dlx_word;
    variable IR  : dlx_word;
    variable IAR : dlx_word;
    variable ICR : dlx_word;
    variable TBR : dlx_word;    
    variable MAR : dlx_word;
    variable MDR : dlx_word;
    
    alias IR_opcode   : dlx_opcode is IR(0 to 5);
    alias IR_rr_func  : dlx_rr_func is IR(26 to 31);
    alias IR_rs1      : dlx_reg_addr is IR(6 to 10);
    alias IR_rs2      : dlx_reg_addr is IR(11 to 15);
    alias IR_rd_Itype : dlx_reg_addr is IR(11 to 15);
    alias IR_rd_Rtype : dlx_reg_addr is IR(16 to 20);
    alias IR_immed16  : dlx_immed16 is IR(16 to 31);
    alias IR_immed26  : dlx_immed26 is IR(6 to 31);

    variable rs1, rs2 : reg_index;
    variable rd_Itype : reg_index;
    variable rd_Rtype : reg_index;

    --
    -- further variables
    --
    constant PC_incr : dlx_word := To_StdLogicVector(X"0000_0004");

    variable L : line;
    variable tmp_word : dlx_word;                  -- temporary storage

  -----------------------------------------------------------------
  -- begin of the interpreter process
  -----------------------------------------------------------------
    
  begin 
    --
    -- normal fetch-decode-execute loop with interrupt handling
    -- (loop is only be exited in case of asynchronous reset or
    --  exceptions during execution)
    --
    fetch_dec_exc: loop 
      --
      -- check for active reset signal and exit the loop in this case 
      --
      exit fetch_dec_exc when reset = '1';
      --
      -- fetch next instruction
      --
      if DEBUG then
        write(L, string'(">>> fetching instruction from address: "));
	write(L, PC, X, up);
        writeline(output, L);
      end if;
      
      bus_read(address => PC, data_width => word, data => IR,
               phi1 => phi1, phi2 => phi2, addr_bus => a_bus, data_bus => d_bus,
 	       reset => reset, ready => ready, rw => rw, enable => enable,
 	       tpd_behav=> tpd_behav);
      exit fetch_dec_exc when reset = '1';
      --
      -- no interrupt detected => normal decode and execution
      -- increment the PC to point to the following instruction
      --
      if DEBUG then
	write(L, string'(">>> incrementing PC..."));
	writeline(output, L);
      end if;

      sv_add(PC, PC_incr, tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
      PC := tmp_word;
      --
      -- decode the instruction
      --
      if DEBUG then
	write(L, string'(">>> decoding instruction..."));
	writeline(output, L);
	write_instr(L, IR);
	writeline(output, L);
      end if;

      rs1            := sv_to_natural(IR_rs1);
      rs2            := sv_to_natural(IR_rs2);
      rd_Itype       := sv_to_natural(IR_rd_Itype);
      rd_Rtype       := sv_to_natural(IR_rd_Rtype);

      --
      -- (no) execute
      --
      if DEBUG then
	write(L, string'(">>> end of execution"));
	writeline(output, L);
      end if;
      --
      -- loop is only exited when reset is active
      --      
    end loop;

    --
    -- check for illegal exiting of fetch_decode_execute loop (assertion)
    --
    assert reset = '1'
      report "DLX: main loop exited unexpectedly"
      severity failure;
      
    if reset = '1' then
      --
      -- reset the processor<check for
      -- (no reset for GP_REG(1 to 31) )
      -- (no reset neccessary for IR, IAR, MAR, MDR)
      --
      if DEBUG then
        write(L, string'(">>> reset detected, resetting processor..."));
        writeline(output, L);
      end if;
	
      d_bus <= (others => 'Z');
      enable <= "0000";
      rw <= '0';
      GP_REG(0) := (others => '0');
      PC  := (others => '0');
      ICR := To_StdLogicVector(X"0000_0001");
      --
      -- wait for inactivation of reset signal
      --
      wait until phi2 = '0' and reset = '0';
    end if; 

    --
    -- process interpreter now starts again from beginning
    --
  end process interpreter;

end behaviour_1;


