--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  Package body for the procedures used in the behavioural
--  descriptions of DLX processor within the exercises
--  (for the architectures behaviour_1, behaviour_2, behaviour_3)
--  
--  (file behav_procedures_123-body.vhd)
--------------------------------------------------------------------------

PACKAGE BODY behav_procedures_123 IS

  -----------------------------------------------------------------
  -- procedure for reading from the data bus 
  -----------------------------------------------------------------
    PROCEDURE bus_read (address    : IN dlx_address;
                        data_width : IN mem_ACCESS_TYPE;
			data       : OUT dlx_word;
			--
			signal phi1, phi2 : IN std_logic;
                        signal addr_bus   : OUT dlx_address;
			signal data_bus   : INOUT dlx_word;
			signal reset      : IN std_logic;
			signal ready      : IN std_logic;
			signal rw         : OUT std_logic;
			signal enable     : OUT dlx_nibble;
			--
			tpd_behav : IN time) IS
    BEGIN
      --
      -- place the address on the address bus after the rising edge of phi2
      --
      WAIT UNTIL phi2 = '1';
      IF reset = '1' THEN
 	RETURN;
      END IF;      
      addr_bus <= address AFTER tpd_behav; 
      --
      -- set ENABLE and RW after the rising edge of phi1
      --
      WAIT UNTIL phi1 = '1';
      IF reset = '1' THEN 
        RETURN;
      END IF;
      rw <= '1' AFTER tpd_behav;
      case data_width is
	when word   => enable <= "1111" after tpd_behav;
	when half   =>
	  case address(1) is
	    when '0' => enable <= "1100" after tpd_behav;
	    when others => enable <= "0011" after tpd_behav;
 	  end case;
	when others =>
 	  case address(1 downto 0) is
	    when "00" => enable <= "1000" after tpd_behav;
	    when "01" => enable <= "0100" after tpd_behav;
	    when "10" => enable <= "0010" after tpd_behav;
	    when others => enable <= "0001" after tpd_behav;
  	  end case;
      end case ;
      --
      -- wait until data ready while checking for interrupts and reset
      --
      LOOP 
	wait until phi2 = '1';                   -- wait until rising edge of phi2
        WAIT UNTIL phi2 = '0';                   -- wait until falling edge of phi2
        EXIT WHEN ready = '1' OR reset = '1';
      END LOOP;
      data := TO_UX01(data_bus);                 -- register content can't be 'Z'
      --
      -- inactivate lines after rising edge of phi1
      --
      WAIT UNTIL phi1 = '1';
      IF reset = '1' THEN 
        RETURN;
      END IF;
      data_bus <= (OTHERS => 'Z') AFTER tpd_behav;
      enable <= "0000" after tpd_behav;
      rw <= '0' AFTER tpd_behav;
    END bus_read;

  -----------------------------------------------------------------
  -- procedure for writing on the data bus 
  -----------------------------------------------------------------
    procedure bus_write (address    : IN  dlx_address;
                         data_width : IN  mem_access_type;
                         data       : IN  dlx_word;
			 --
			 signal phi1, phi2 : IN std_logic;
                         signal addr_bus   : OUT dlx_address;
			 signal data_bus   : INOUT dlx_word;
			 signal reset      : IN std_logic;
			 signal ready      : IN std_logic;
			 signal rw         : OUT std_logic;
			 signal enable     : OUT dlx_nibble;
			 --
			 tpd_behav : IN time) IS
    begin
      --
      -- place the address on the address bus after the rising edge of phi2
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;      
      addr_bus <= address after tpd_behav; 
      --
      -- place data on the data bus after rising edge of phi2 
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;
      data_bus <= data after tpd_behav;
      --
      -- set ENABLE and RW after the rising edge of phi1 & check for interrupts
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      rw <= '0' after tpd_behav;
      case data_width is
	when word   => enable <= "1111" after tpd_behav;
	when half   =>
	  case address(1) is
	    when '0' => enable <= "1100" after tpd_behav;
	    when others => enable <= "0011" after tpd_behav;
 	  end case;
	when others =>
 	  case address(1 downto 0) is
	    when "00" => enable <= "1000" after tpd_behav;
	    when "01" => enable <= "0100" after tpd_behav;
	    when "10" => enable <= "0010" after tpd_behav;
	    when others => enable <= "0001" after tpd_behav;
  	  end case;
      end case ;
      --
      -- wait until data accepted while checking for interrupts and reset
      --
      LOOP 
	wait until phi2 = '1';                   -- wait until rising edge of phi2
        WAIT UNTIL phi2 = '0';                   -- wait until falling edge of phi2
        EXIT WHEN ready = '1' OR reset = '1';
      END LOOP;
      --
      -- inactivate lines after rising edge of phi1
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      data_bus <= (others => 'Z') after tpd_behav;
      enable <= "0000" after tpd_behav;
      rw <= '0' after tpd_behav;
    end bus_write;

END behav_procedures_123;




