-- ****************************************************************************
-- *                                                                          *
-- *     THIS PACKAGE CONSISTS OF THE FUNCTIONS FOR ANALOG TO DIGITAL AND     *
-- *                      DIGITAL TO ANALOG CONVERSIONS                       *
-- *                                                                          *
-- ****************************************************************************

use STD.TEXTIO.all;
use work.COMPUTE_PACK.all;
use work.IO_PACK.all;

-- ****************************************************************************
-- *									      *
-- *               PACKAGE INTERFACE DECLARATIONS                             *
-- *									      *
-- ****************************************************************************

 package INTERFACE_PACK is 

 subtype SWITCH_LEVEL is character range '-' to 'Z';
 type SWITCH_LEVEL_VECTOR is array (natural range<>) of SWITCH_LEVEL;
 type logic_array is array (natural range<>) of SWITCH_LEVEL;
 type time_array is array(natural range<>) of time;
 type volt_array is array(natural range<>) of real;

 function threshold(v_output : in real) return SWITCH_LEVEL;

 function threshold_1(v_output : in real) return SWITCH_LEVEL;
 
 function high_imped (current_out1 : in real) return SWITCH_LEVEL;

 function digi2ana_inv (LOGIC_IN : in SWITCH_LEVEL; t_real : in real) 
                        return real;

 function digi2ana (LOGIC_IN : in SWITCH_LEVEL; t_real : in real) return real;

 function digi2ana_delay (LOGIC_IN : in SWITCH_LEVEL; t_real : in real; 
                          delta_real : real) return real;

 function digi2ana_delay_inv (LOGIC_IN : in SWITCH_LEVEL; t_real : in real;
                              delta_real : real) return real;

 function INT_TO_REAL (A : in integer) return real;

 function delta (voltarray : in volt_array; hi_cutoff : in real;
                 lo_cutoff : in real) return integer;

 function error1 (anaout : in real) return real;

-- procedure differ(t : in TIME; 
--                  stop_after : in TIME; LOGIC_OUT : in SWITCH_LEVEL);

end INTERFACE_PACK;


-- *****************************************************************************
-- *									       *
-- *              PACKAGE INTERFACE BODY                                       *-- *									       *-- *****************************************************************************

package body INTERFACE_PACK is

  function threshold (v_output : in real) return SWITCH_LEVEL 
  is

-- This function checks for the cutoff limits in the incoming analog voltage
-- and outputs the DIGITAL values in three levels '0','1' & 'X'.

      constant high    : real := 3.7;
      constant low     : real := 1.7;
      constant weak0L   : real := -1.9;
      constant weak0H   : real := -1.7;
      constant weak1L   : real := -1.045;
      constant weak1H   : real := -0.8;
      variable result  : SWITCH_LEVEL;
      variable k       : LINE;

  file FILE2 : TEXT is out "STD_OUTPUT";

  begin

     if (v_output > 0.0 and v_output <= low) then
        result := '0';                                   -- '0' - Forcing 0
     elsif (v_output > high and v_output <= 5.0) then
        result := '1';                                   -- '1' - Forcing 1
     elsif (v_output > low and v_output < high) then 
        result := 'X';                                   -- 'X' - Forcing 0 or 1
     elsif (v_output <= weak0H and v_output >= weak0L)then
        result := 'L';                                   -- 'L' - Weak 0
     elsif (v_output <= weak1H and v_output >= weak1L)then
        result := 'H';                                   -- 'H' - Weak 1
     elsif (v_output < weak1L and v_output > weak0H) then
        result := 'W';                                   -- 'W' - Weak 0 or 1
     elsif (v_output < 0.0 and v_output > weak1H) then
        result := '-';                                   -- 'U' - Uninitialised
     elsif v_output < weak0L then
        result := 'U';                                   -- '-' - Don't Care	
     end if;
     return result;

  end threshold;

-- *****************************************************************************

  function threshold_1 (v_output : in real) return SWITCH_LEVEL
  is

-- This function checks for the cutoff limits in the incoming analog voltage
-- and outputs the DIGITAL values in three levels '0' & '1'.

    constant high    : real := 3.7;
    variable result  : SWITCH_LEVEL;
    variable k       : LINE;

  file FILE2 : TEXT is out "STD_OUTPUT";

  begin

     if v_output >= high then
        result := '1';
     else
        result := '0';
     end if;
     return result;

  end threshold_1;

-- *****************************************************************************  

  function high_imped (current_out1 : in real) return SWITCH_LEVEL
  is

-- This function outputs the logic value 'Z' taking the current (Ids) into 
-- consideration.

    constant i_min  : real := 3.0e-04;
    variable result : SWITCH_LEVEL;
    variable k      : LINE;

  file FILE2 : TEXT is out "STD_OUTPUT";

  begin

      if current_out1 <= i_min then
         result := 'Z';
      end if;
      return result;

  end high_imped;
     

-- *****************************************************************************

function digi2ana(LOGIC_IN : in SWITCH_LEVEL; t_real : in real) return real 
is

-- This function converts rising digital voltage input, into analog
-- value.

     constant E          : real := 2.7182;
     variable ana_value  : real;
     constant vdd        : real := 5.0;
     constant vss        : real := 0.1;
     constant weak0      : real := -1.8;
     constant weak1      : real := -0.92;
     constant damp       : real := 0.15;
     variable prod_damp  : real;
     variable temp1      : real;
     variable temp2      : real;
     variable temp3      : real;
     variable temp4      : real;
     variable temp5      : real;
     variable temp6      : real;

file FILE2 : TEXT is out "STD_OUTPUT";

begin

   prod_damp := damp * t_real;
   if LOGIC_IN = '1' then
      temp1 := vdd /POWER(E,prod_damp);
      ana_value := vdd - temp1;
   elsif LOGIC_IN = '0' then
      temp2 := vss/POWER(E,prod_damp);
      ana_value := vss - temp2; 
   elsif LOGIC_IN = 'X' then
      temp3 := ((vdd + vss) / 2.0) / POWER(E,prod_damp);
      ana_value := ((vdd + vss) / 2.0) - temp3;
   elsif LOGIC_IN = 'H' then
      temp4 := weak1/POWER(E,prod_damp);
      ana_value := weak1 - temp4;
   elsif LOGIC_IN = 'L' then
      temp5 := weak0/POWER(E,prod_damp);
      ana_value := weak0 - temp5;
   elsif LOGIC_IN = 'W' then
      temp6 := ((weak1 + weak0) / 2.0) / POWER(E,prod_damp);
      ana_value := ((weak1 + weak0) / 2.0) - temp6;
   end if;
   return ana_value;

end digi2ana;

-- *****************************************************************************

function digi2ana_inv (LOGIC_IN : in SWITCH_LEVEL; t_real : in real) 
                       return real is

-- This function converts falling digital voltage input, into analog
-- value.

     constant E          : real := 2.7182;
     variable ana_value  : real;
     constant vdd        : real := 5.0;
     constant vss        : real := 0.1;
     constant weak0      : real := -1.8;
     constant weak1      : real := -0.92;
     constant damp       : real := 0.15;
     variable prod_damp  : real;


file FILE2 : TEXT is out "STD_OUTPUT";

begin

   prod_damp := damp * t_real;
   if LOGIC_IN = '1' then
      ana_value := vdd / POWER(E,prod_damp);
   elsif LOGIC_IN = '0' then
      ana_value := vss / POWER(E,prod_damp); 
   elsif LOGIC_IN = 'X' then
       ana_value := ((vdd + vss) / 2.0) / POWER(E,prod_damp);
   elsif LOGIC_IN = 'H' then
      ana_value := weak1 / POWER(E,prod_damp);
   elsif LOGIC_IN = 'L' then
      ana_value := weak0 / POWER(E,prod_damp);
   elsif LOGIC_IN = 'W' then
       ana_value := ((weak1 + weak0) / 2.0) / POWER(E,prod_damp);
   end if;
   return ana_value;

end digi2ana_inv;


-- ****************************************************************************    

function digi2ana_delay(LOGIC_IN : in SWITCH_LEVEL; t_real : in real;
                         delta_real : real) return real is

-- This function converts rising digital input, with delta switching
-- delay into analog.

     constant E          : real := 2.7182;
     variable ana_value  : real;
     constant vdd        : real := 5.0;
     constant vss        : real := 0.1;
     constant weak0      : real := -1.8;
     constant weak1      : real := -0.92;
     constant damp       : real := 0.15;
     variable prod_damp  : real;
     variable prod_damp1 : real;
     variable temp1      : real;
     variable temp2      : real;
     variable temp3      : real;
     variable temp4      : real;
     variable temp5      : real;
     variable temp6      : real;
     variable K          : LINE;

file FILE2 : TEXT is out "STD_OUTPUT";

begin

   prod_damp := damp * t_real;
   prod_damp1:= damp * (t_real * delta_real);
   if LOGIC_IN = '1' then
      temp1 := vdd / POWER(E,prod_damp);
      ana_value := vdd - temp1;
   elsif LOGIC_IN = '0' then
      temp2 := vss / POWER(E,prod_damp); 
      ana_value := vss - temp2; 
   elsif LOGIC_IN = 'X' then
      if delta_real <= 10.0 then
         temp3 := ((vdd + vss) / 2.0) / POWER(E,prod_damp);
      else
         temp3 := ((vdd + vss) / 2.0) / POWER(E,prod_damp);
      end if;
      ana_value := ((vdd + vss) / 2.0) - temp3;
   elsif LOGIC_IN = 'H' then
      temp4 := weak1/POWER(E,prod_damp);
      ana_value := weak1 - temp4;
   elsif LOGIC_IN = 'L' then
      temp5 := weak0/POWER(E,prod_damp);
      ana_value := weak0 - temp5;
   elsif LOGIC_IN = 'W' then
      if delta_real <= 10.0 then
         temp6 := ((weak1 + weak0) / 2.0) / POWER(E,prod_damp);
      else
         temp6 := ((weak1 + weak0) / 2.0) / POWER(E,prod_damp);
      end if;
      ana_value := ((weak1 + weak0) / 2.0) - temp6;
   end if;
   return ana_value;

end digi2ana_delay;

-- *****************************************************************************

function digi2ana_delay_inv(LOGIC_IN : in SWITCH_LEVEL; t_real : in real;
                              delta_real : real) return real is

-- This function converts falling digital input with, delta switching
-- delay into analog.

     constant E          : real := 2.7182;
     variable ana_value  : real;
     constant vdd        : real := 5.0;
     constant vss        : real := 0.1;
     constant weak0      : real := -1.8;
     constant weak1      : real := -0.92;
     constant damp       : real := 0.15;
     variable prod_damp  : real;
     variable prod_damp1 : real;

file FILE2 : TEXT is out "STD_OUTPUT";

begin

   prod_damp := damp * t_real;
   prod_damp1:= damp *(t_real * delta_real);
   if LOGIC_IN = '1' then
     ana_value := vdd / POWER(E,prod_damp);
   elsif LOGIC_IN = '0' then
     ana_value := vss / POWER(E,prod_damp); 
   elsif LOGIC_IN = 'X' then
     if delta_real <= 10.0 then
       ana_value := ((vdd + vss) / 2.0) / POWER(E,prod_damp);
     else
       ana_value := ((vdd + vss) / 2.0) / POWER(E,prod_damp);
     end if; 
   elsif LOGIC_IN = 'H' then
     ana_value := weak1 / POWER(E,prod_damp);
   elsif LOGIC_IN = 'L' then
     ana_value := weak0 / POWER(E,prod_damp);
   elsif LOGIC_IN = 'W' then
     if delta_real <= 10.0 then
       ana_value := ((weak1 + weak0) / 2.0) / POWER(E,prod_damp);
     else
       ana_value := ((weak1 + weak0) / 2.0) / POWER(E,prod_damp);
     end if;
   end if;
   return ana_value;

end digi2ana_delay_inv;


-- *****************************************************************************

 function delta (voltarray : in volt_array; hi_cutoff : in real;
                 lo_cutoff : in real) return integer is

      variable i  : integer := 1;
      variable d1 : integer := 0;
      variable d2 : integer := 0;
      variable d  : integer := 0;
      variable K  : line;

   begin

     loop1:
       for i in 1 to 100 loop
         if voltarray(i) <= hi_cutoff then
           d1 := i;
           exit loop1;
         end if;
     end loop loop1;
     loop2:
       for i in 1 to 100 loop
         if voltarray(i) <= lo_cutoff then
           d2 := i;
           exit loop2;
         end if;
     end loop loop2;
     d := abs(d1 - d2);
     return d;

 end delta;

-- *****************************************************************************

 function error1 (anaout : in real) return real is

      constant ref_volt : real := 5.0;
      variable error_value : real;


   begin
  
      error_value := ref_volt - anaout;
      return error_value;

 end error1;

      
-- *****************************************************************************

--  procedure differ (t : in TIME;
--                    stop_after : in TIME; LOGIC_OUT : in SWITCH_LEVEL) is 

--    variable t1          : TIME := 0 ns;
--    variable d           : integer;
--    variable d1          : integer;
--    variable d2          : integer;
--    variable tdiff       : integer;
--    variable k           : LINE;
--    variable j           : integer := 1;
--    variable logic_value : logic_array(1 to 100);
--    variable t_real      : real;

--  file FILE2 : TEXT is out "STD_OUTPUT";

--  begin
    
--    if t < stop_after then
--      loop
--       logic_value(j) := LOGIC_OUT;
--       write(k,"T = ");
--       write(k,t);
--       write(k,"    ");
--       write(k,"LOGIC = ");
--       write(k,logic_value(j));
--       write(k,"   ");
--       write(k,"j = ");
--       write(k,j);
--       write(k,"    ");
--       j := j + 1;
--       t1 := t1 + 1 ns;
--       exit when t = stop_after;
--     end loop;
--    end if;
--    if logic_value(j) = '1' then
--      j := j + 1;
--      loop1:
--      loop
--        write(k,"I am in loop1");
--        write(k,"    ");
--        if logic_value(j) = '0' then
--          d1 := j;
--          exit loop1;
--        elsif logic_value(j) = 'X' then
--          d1 := j;
--          if logic_value(j) /= '1' and logic_value(j) /= '0' and j < 100 then
--            loop
--              j := j + 1;
--            end loop;
--            d2 := j;
--          end if;
--        end if;
--      end loop loop1;
--    elsif logic_value(j) = '0' then
--      j := j + 1;
--      loop2:
--      loop
--        if logic_value(j) ='1' then
--          d2 := j;
--          exit loop2;
--        elsif logic_value(j) ='X' then
--          d1 := j;
--          if logic_value(j) /= '1' and logic_value(j) /= '0' and j < 100 then
--            loop
--              j := j + 1;
--            end loop;
--            d2 := j;
--          end if;
--        end if;
--      end loop loop2;
--    end if;
--    tdiff := abs(d1 - d2);
--    write(k,"tdiff = ");
--    write(k,tdiff);
--    write(k,"    ");   
--    writeline(OUTPUT,k);
--    return;

--  end differ;

-- *****************************************************************************
 
  function INT_TO_REAL(A: in integer) return REAL
  is
    variable x : integer;
    variable y : integer;
    variable B : integer := 0;
    variable R : real := 0.00000;
    variable F : real := 1.00000;
    variable T_REAL : real := 0.00000;
    variable FACT: real:=1.0;
    variable k : line;

file FILE2 : TEXT is out "STD_OUTPUT";

  begin

    x := A;                               --Input  Time   value   A
    y := A;                               --Input  Time   value   A
    if x >= 10  then
      FACT := 1.0;
    elsif x >= 1 and x < 10  then
      FACT:=0.01;
      x := 100 * x;
      y := 100 * y;
    end if;
    loop0:
    loop
      x := (x/10 ) * 1 ;            --Gelet the last bit of x.
      if x = 0  and y = 0  then     --No more bit left.
        exit loop0;
      end if;
      B := (y - x * 10) / 1  ;        --Extract the GeleteG bit
      y := (y/10 ) * 1 ;              --Gelet the last bit  for
      if B = 0 then                     --CSTEPVange  integer  bit  B
        R :=    0.00000;
      elsif B = 1 then
        R :=  1.00000;
      elsif B = 2 then
        R :=  2.00000;
      elsif B = 3 then
        R :=  3.00000;
      elsif B = 4 then
        R :=  4.00000;
      elsif B = 5 then
        R :=  5.00000;
      elsif B = 6 then
        R :=  6.00000;
      elsif B = 7 then
        R :=  7.00000;
      elsif B = 8 then
        R :=  8.00000;
      elsif B = 9 then
        R :=  9.00000;
      end if;
      T_REAL := T_REAL + R * F;         --Get back the input time
      F := F * 10.00000;                --value but is a floating
    end loop loop0;                         --STEPV
    T_REAL := FACT*T_REAL;                 --unit on it.
  return T_REAL;
  end INT_TO_REAL;

-- *****************************************************************************

end INTERFACE_PACK;
