--++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
-- 
-- numeric_shift.vhd file is a package containing the shift and rotate
-- operators for the numeric_bit package. This is a first draft which is
-- being circulated for testing and change recommendations.
--
-- I have included the signatures in the package header and body, for
-- reference purposes. We should decide on a uniform representation
-- for the final documentation.
-- 
-- Rob Anderson, Dec. 1 1993
-- internet correspondence: rea@netcom.com
--
-- changes Dec. 20:
--         Based on reviewing the '92 LRM and the high level specification,
--         I have revised the SLA an SRA functions. 
--=============================================================================+



package NUMERIC_shifts is

   -- Vector types

   type UNSIGNED is array ( NATURAL range <> ) of BIT;
   type SIGNED is array ( NATURAL range <> ) of BIT;

   -- Standard null range array constants

   constant nau : UNSIGNED (0 downto 1) := (others => '0');
   constant nas : SIGNED (0 downto 1) := (others => '0');


--==============================================================================
-- Signature:    1027  f  sll  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function sll ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1028  f  srl  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function srl ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1029  f  sla  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function sla ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1030  f  sra  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function sra ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1031  f  rol  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function rol ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1032  f  ror  UBV n c INT x c UBV n V         
--------------------------------------------------------------------------------
function ror ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1033  f  sll  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function sll ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1034  f  srl  BV2 n c INT x c BV2 n V 
--------------------------------------------------------------------------------
function srl ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1035  f  sla  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function sla ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1036  f  sra  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function sra ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1037  f  rol  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function rol ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1038  f  ror  BV2 n c INT x c BV2 n V     
--------------------------------------------------------------------------------
function ror ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1019  f  sll  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function sll ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1020  f  srl  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function srl ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1021  f  sla  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function sla ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1022  f  sra  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function sra ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1023  f  rol  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function rol ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1024  f  ror  BV n c INT x c BV n V   
--------------------------------------------------------------------------------
function ror ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR;
--------------------------------------------------------------------------------


end NUMERIC_shifts;

--*******************************************************************************
--*******************************************************************************
--*******************************************************************************

package body numeric_shifts is


--==============================================================================
-- Signature:    1027  f  sll  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function sll ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : UNSIGNED(arg_l downto 0) is arg;
  variable sll_return : UNSIGNED(arg_l downto 0) := (others=>'0');
begin
  if count < 0 then sll_return:=srl(arg,-count);
  elsif count <= arg_l then
    sll_return(arg_l downto count):=xarg(arg_l-count downto 0);
    end if;
  return sll_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1028  f  srl  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function srl ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : unsigned(0 to arg_l) is arg;
  variable srl_return : unsigned(0 to arg_l) := (others=>'0');
begin
  if count < 0 then srl_return:=sll(arg,-count);
  elsif count <= arg_l then
    srl_return(count to arg_l):=xarg(0 to arg_l-count);
    end if;
  return srl_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1029  f  sla  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function sla ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : unsigned(arg_l downto 0) is arg;
  variable sla_return : unsigned(arg_l downto 0) := (others=>xarg(0));
begin
  if count < 0 then sla_return:=sra(arg,-count);
  elsif count <= arg_l then
    sla_return(arg_l downto count):=xarg(arg_l-count downto 0);
    end if;
  return sla_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1030  f  sra  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function sra ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : unsigned(0 to arg_l) is arg;
  variable sra_return : unsigned(0 to arg_l) := (others=>xarg(0));
begin
  if count < 0 then sra_return:=sla(arg,-count);
  elsif count <= arg_l then
    sra_return(count to arg_l):=xarg(0 to arg_l-count);
    end if;
  return sra_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1031  f  rol  UBV n c INT x c UBV n V
--------------------------------------------------------------------------------
function rol ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : unsigned(arg_l downto 0) is arg;
  variable rol_return : unsigned(arg_l downto 0) := xarg;
  variable countm : integer;
begin
  countm := count mod (arg_l + 1);
  if countm < 0 then rol_return:=ror(arg,-countm);
  elsif countm /= 0 then
    rol_return(arg_l downto countm):=xarg(arg_l-countm downto 0);
    rol_return(countm-1 downto 0):=xarg(arg_l downto arg_l-countm+1);
    end if;
  return rol_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1032  f  ror  UBV n c INT x c UBV n V         
--------------------------------------------------------------------------------
function ror ( CONSTANT  arg  : UNSIGNED;
               CONSTANT count : INTEGER ) return UNSIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : unsigned(0 to arg_l) is arg;
  variable ror_return : unsigned(0 to arg_l) := xarg;
  variable countm : integer;
begin
  countm := count mod (arg_l + 1);
  if countm < 0 then ror_return:=rol(arg,-countm);
  elsif countm /= 0 then
    ror_return(countm to arg_l):=xarg(0 to arg_l-countm);
    ror_return(0 to countm-1):=xarg(arg_l-countm+1 to arg_l);
    end if;
  return ror_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1033  f  sll  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function sll ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : signed(arg_l downto 0) is arg;
  variable sll_return : signed(arg_l downto 0) := (others=>'0');
begin
  if count < 0 then sll_return:=srl(arg,-count);
  elsif count <= arg_l then
    sll_return(arg_l downto count):=xarg(arg_l-count downto 0);
    end if;
  return sll_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1034  f  srl  BV2 n c INT x c BV2 n V 
--------------------------------------------------------------------------------
function srl ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : signed(0 to arg_l) is arg;
  variable srl_return : signed(0 to arg_l) := (others=>'0');
begin
  if count < 0 then srl_return:=sll(arg,-count);
  elsif count <= arg_l then
    srl_return(count to arg_l):=xarg(0 to arg_l-count);
    end if;
  return srl_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1035  f  sla  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function sla ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : signed(arg_l downto 0) is arg;
  variable sla_return : signed(arg_l downto 0) := (others=>xarg(0));
begin
  if count < 0 then sla_return:=sra(arg,-count);
  elsif count <= arg_l then
    sla_return(arg_l downto count):=xarg(arg_l-count downto 0);
    end if;
  return sla_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1036  f  sra  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function sra ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : signed(0 to arg_l) is arg;
  variable sra_return : signed(0 to arg_l) := (others=>xarg(0));
begin
  if count < 0 then sra_return:=sla(arg,-count);
  elsif count <= arg_l then
    sra_return(count to arg_l):=xarg(0 to arg_l-count);
    end if;
  return sra_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1037  f  rol  BV2 n c INT x c BV2 n V
--------------------------------------------------------------------------------
function rol ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : signed(arg_l downto 0) is arg;
  variable rol_return : signed(arg_l downto 0) := xarg;
  variable countm : integer;
begin
  countm := count mod (arg_l + 1);
  if countm < 0 then rol_return:=ror(arg,-countm);
  elsif countm /= 0 then
    rol_return(arg_l downto countm):=xarg(arg_l-countm downto 0);
    rol_return(countm-1 downto 0):=xarg(arg_l downto arg_l-countm+1);
    end if;
  return rol_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1038  f  ror  BV2 n c INT x c BV2 n V     
--------------------------------------------------------------------------------
function ror ( CONSTANT  arg  : SIGNED;
               CONSTANT count : INTEGER ) return SIGNED is
  constant arg_l:integer := arg'length-1;
  alias xarg : signed(0 to arg_l) is arg;
  variable ror_return : signed(0 to arg_l) := xarg;
  variable countm : integer;
begin
  countm := count mod (arg_l + 1);
  if countm < 0 then ror_return:=rol(arg,-countm);
  elsif countm /= 0 then
    ror_return(countm to arg_l):=xarg(0 to arg_l-countm);
    ror_return(0 to countm-1):=xarg(arg_l-countm+1 to arg_l);
    end if;
  return ror_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1019  f  sll  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function sll ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR is
  constant arg_l:integer := arg'length-1;
  alias xarg : bit_vector(arg_l downto 0) is arg;
  variable sll_return : bit_vector(arg_l downto 0) := (others=>'0');
begin
  if count < 0 then sll_return:=srl(arg,-count);
  elsif count <= arg_l then
    sll_return(arg_l downto count):=xarg(arg_l-count downto 0);
    end if;
  return sll_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1020  f  srl  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function srl ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR is
  constant arg_l:integer := arg'length-1;
  alias xarg : bit_vector(0 to arg_l) is arg;
  variable srl_return : bit_vector(0 to arg_l) := (others=>'0');
begin
  if count < 0 then srl_return:=sll(arg,-count);
  elsif count <= arg_l then
    srl_return(count to arg_l):=xarg(0 to arg_l-count);
    end if;
  return srl_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1021  f  sla  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function sla ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR is
  constant arg_l:integer := arg'length-1;
  alias xarg : bit_vector(arg_l downto 0) is arg;
  variable sla_return : bit_vector(arg_l downto 0) := (others=>xarg(0));
begin
  if count < 0 then sla_return:=sra(arg,-count);
  elsif count <= arg_l then
    sla_return(arg_l downto count):=xarg(arg_l-count downto 0);
    end if;
  return sla_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1022  f  sra  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function sra ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR is
  constant arg_l:integer := arg'length-1;
  alias xarg : bit_vector(0 to arg_l) is arg;
  variable sra_return : bit_vector(0 to arg_l) := (others=>xarg(0));
begin
  if count < 0 then sra_return:=sla(arg,-count);
  elsif count <= arg_l then
    sra_return(count to arg_l):=xarg(0 to arg_l-count);
    end if;
  return sra_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1023  f  rol  BV n c INT x c BV n V
--------------------------------------------------------------------------------
function rol ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR is
  constant arg_l:integer := arg'length-1;
  alias xarg : bit_vector(arg_l downto 0) is arg;
  variable rol_return : bit_vector(arg_l downto 0) := xarg;
  variable countm : integer;
begin
  countm := count mod (arg_l + 1);
  if countm < 0 then rol_return:=ror(arg,-countm);
  elsif countm /= 0 then
    rol_return(arg_l downto countm):=xarg(arg_l-countm downto 0);
    rol_return(countm-1 downto 0):=xarg(arg_l downto arg_l-countm+1);
    end if;
  return rol_return;
end;
--------------------------------------------------------------------------------

--==============================================================================
-- Signature:    1024  f  ror  BV n c INT x c BV n V   
--------------------------------------------------------------------------------
function ror ( CONSTANT  arg  : BIT_VECTOR;
               CONSTANT count : INTEGER ) return BIT_VECTOR is
  constant arg_l:integer := arg'length-1;
  alias xarg : bit_vector(0 to arg_l) is arg;
  variable ror_return : bit_vector(0 to arg_l) := xarg;
  variable countm : integer;
begin
  countm := count mod (arg_l + 1);
  if countm < 0 then ror_return:=rol(arg,-countm);
  elsif countm /= 0 then
    ror_return(countm to arg_l):=xarg(0 to arg_l-countm);
    ror_return(0 to countm-1):=xarg(arg_l-countm+1 to arg_l);
    end if;
  return ror_return;
end;
--------------------------------------------------------------------------------

end numeric_shifts;
